﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.UI;

public class MessageBoxManager : MonoBehaviour
{
    private enum ButtonEnum
    {
        LeftButton,
        CentreButton,
        RightButton
    }

    //The root MessageBox game object
    public GameObject MessageBox;

    //Components of the MessageBox
    private GameObject _Content;
    private TMP_Text _TitleText;
    private TMP_Text _MessageText;
    private Button _LeftButton;
    private Button _CentreButton;
    private Button _RightButton;

    private int _ButtonsCount;
    public int SelectedButton { get; private set; } = -1;
    public bool ShowingMessageBox { get; private set; } = false;
    private GameObject _PreviousFocusedGameObject;

    private TaskCompletionSource<bool> ClosedTask;  //Task to allow waiting of page to close

    private static MessageBoxManager _Instance;

    public static MessageBoxManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<MessageBoxManager>();
            }

            return _Instance;
        }
    }

    private void Start()
    {
        //Get all the MessageBox components
        _Content = MessageBox.FindChild("Content");

        _TitleText = _Content.FindChild("BackgroundPanel").FindChild("Banner").FindChild("TitleText").GetComponent<TMP_Text>();
        _MessageText = _Content.FindChild("BackgroundPanel").FindChild("MessageText").GetComponent<TMP_Text>();
        _LeftButton = _Content.FindChild("BackgroundPanel").FindChild("Buttons").FindChild("LeftButton").GetComponent<Button>();
        _CentreButton = _Content.FindChild("BackgroundPanel").FindChild("Buttons").FindChild("CentreButton").GetComponent<Button>();
        _RightButton = _Content.FindChild("BackgroundPanel").FindChild("Buttons").FindChild("RightButton").GetComponent<Button>();

        //Add appropriate listeners to the MessageBox button click events
        _LeftButton.onClick.AddListener(delegate { OnButtonSelected(ButtonEnum.LeftButton); });
        _CentreButton.onClick.AddListener(delegate { OnButtonSelected(ButtonEnum.CentreButton); });
        _RightButton.onClick.AddListener(delegate { OnButtonSelected(ButtonEnum.RightButton); });

        InputManager.Instance.InputStateChanged += OnInputStateChanged;
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (e.NewState == Constants.InputState.Controller && !e.Handled)
        {
            if (ShowingMessageBox)
            {
                EventSystem.current.SetSelectedGameObject(null);

                switch (_ButtonsCount)
                {
                    case 1:
                    default:
                        EventSystem.current.SetSelectedGameObject(_CentreButton.gameObject);
                        break;

                    case 2:
                    case 3:
                        EventSystem.current.SetSelectedGameObject(_LeftButton.gameObject);
                        break;
                }

                e.Handled = true;
            }
        }
    }

    /// <summary>
    /// Shows the MessageBox, disabling the current page and returns with the index of the selected button when selected and then re-enables the current page
    /// </summary>
    /// <param name="title">The title text of the MessageBox</param>
    /// <param name="text">The content text of the MessageBox</param>
    /// <param name="buttons">A list of strings for the text of each button</param>
    /// <returns></returns>
    public async Task<int> Show(string title, string text, List<string> buttons)
    {
        //Disable and hide the current page
        PageManager.CurrentPage.PageGameObject.Disable();
        PageManager.CurrentPage.PageGameObject.SetActive(false);

        //Reset values
        ClosedTask = new TaskCompletionSource<bool>();
        SelectedButton = -1;
        _ButtonsCount = buttons.Count;
        _PreviousFocusedGameObject = EventSystem.current.currentSelectedGameObject;


        //Setup the text
        _TitleText.SetText(title);
        _MessageText.SetText(text);

        if (buttons.Count > 0)
        {
            ShowingMessageBox = true;
            _Content.SetActive(true);

            //Setup the buttons and the navigations then show
            if (buttons.Count == 1)
            {
                _LeftButton.gameObject.SetActive(false);
                _RightButton.gameObject.SetActive(false);

                _CentreButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(buttons[0]);
                _CentreButton.gameObject.SetActive(true);

                EventSystem.current.SetSelectedGameObject(_CentreButton.gameObject);
                _CentreButton.OnSelect(null);
            }

            else if (buttons.Count == 2)
            {
                _CentreButton.gameObject.SetActive(false);

                _LeftButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(buttons[0]);
                _RightButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(buttons[1]);
                _LeftButton.gameObject.SetActive(true);
                _RightButton.gameObject.SetActive(true);

                Navigation leftButtonNavigation = _LeftButton.navigation;
                leftButtonNavigation.mode = Navigation.Mode.Explicit;
                leftButtonNavigation.selectOnRight = _RightButton;
                _LeftButton.navigation = leftButtonNavigation;

                Navigation rightButtonNavigation = _RightButton.navigation;
                rightButtonNavigation.mode = Navigation.Mode.Explicit;
                rightButtonNavigation.selectOnLeft = _LeftButton;
                _RightButton.navigation = rightButtonNavigation;

                EventSystem.current.SetSelectedGameObject(_LeftButton.gameObject);
            }

            else
            {
                _LeftButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(buttons[0]);
                _CentreButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(buttons[1]);
                _RightButton.gameObject.FindChild("Text").GetComponent<TMP_Text>().SetText(buttons[2]);
                _LeftButton.gameObject.SetActive(true);
                _CentreButton.gameObject.SetActive(true);
                _RightButton.gameObject.SetActive(true);

                Navigation leftButtonNavigation = _LeftButton.navigation;
                leftButtonNavigation.mode = Navigation.Mode.Explicit;
                leftButtonNavigation.selectOnRight = _CentreButton;
                _LeftButton.navigation = leftButtonNavigation;

                Navigation rightButtonNavigation = _RightButton.navigation;
                rightButtonNavigation.mode = Navigation.Mode.Explicit;
                rightButtonNavigation.selectOnLeft = _CentreButton;
                _RightButton.navigation = rightButtonNavigation;

                Navigation centreButtonNavigation = _CentreButton.navigation;
                centreButtonNavigation.mode = Navigation.Mode.Explicit;
                centreButtonNavigation.selectOnLeft = _LeftButton;
                centreButtonNavigation.selectOnRight = _RightButton;
                _CentreButton.navigation = centreButtonNavigation;

                EventSystem.current.SetSelectedGameObject(_LeftButton.gameObject);
            }

            await ClosedTask.Task;

            //The MessageBox has closed, hide it, show the page again and return the index of the button selected
            _Content.SetActive(false);
            ShowingMessageBox = false;
            PageManager.CurrentPage.PageGameObject.SetActive(true);
            PageManager.CurrentPage.PageGameObject.Enable();
            PageManager.CurrentPage.ReFocus();
            return SelectedButton;
        }

        return -1;
    }

    private void OnButtonSelected(ButtonEnum selectedButton)
    {
        //We selected a button, set the index as appropriate for the buttons visible
        if(selectedButton == ButtonEnum.LeftButton)
        {
            SelectedButton = 0;
        }

        else if(selectedButton == ButtonEnum.CentreButton)
        {
            if(_ButtonsCount == 1)
            {
                SelectedButton = 0;
            }

            else
            {
                SelectedButton = 1;
            }
        }

        else if(selectedButton == ButtonEnum.RightButton)
        {
            if(_ButtonsCount == 2)
            {
                SelectedButton = 1;
            }

            else
            {
                SelectedButton = 2;
            }
        }

        ClosedTask.SetResult(true); //Button selected, time to close
    }
}
